<?php

namespace App\Http\Controllers;

use App\Models\Menu;
use App\Models\MenuType;
use App\Http\Helpers\Helper;
use App\Models\Addon;
use App\Models\Attribute;
use App\Models\MenuAddon;
use App\Models\MenuAttribute;
use App\Models\MenuAttributeValue;
use Illuminate\Http\Request;

class MenuController extends Controller
{
    /**
    * Menu list
    * 
    * @return View
    */
    public function index(Request $request)
    {
        $menus = new Menu;

        if( $request->name )
        $menus = $menus->where('name', 'LIKE', "$request->name%");

        $menus = $menus->orderBy('id', 'desc')->paginate(10);

        return view('menu.index', ['menus' => $menus]);
    }

    /**
    * Get create view
    * 
    * @return View
    */
    public function create()
    {
        $menutypes = MenuType::get();
        $addons = Addon::get();
        $attributes = Attribute::get();

        return view('menu.create', [
            'menutypes' => $menutypes,
            'addons' => $addons,
            'attributes' => $attributes
        ]);
    }

    /**
    * Menu create 
    * 
    * @return View
    */
    public function postCreate(Request $request)
    {
        $input = $request->validate([
            'menu_type_id' => 'sometimes',
            'name' => 'required', 
            'short_description' => 'required', 
            'description' => 'required',
            'ingridients' => 'sometimes',
            'calories' => 'sometimes',
            'allergies' => 'sometimes',
            'image' => 'required|image|max:5000',
            'variations' => 'sometimes',
            'choice_options' => 'sometimes',
            'price' => 'required',
            'discount' => 'sometimes',
            'discount_type' => 'sometimes',
            'available_time_starts' => 'sometimes',
            'available_time_ends' => 'sometimes',
            'status' => 'sometimes'
        ]);

        if ($request->hasFile('image')) {
            $path = $this->uploadImage($request);
            $input['image'] = $path;
        }

        $menu = Menu::create($input);
        $this->createAddon($request, $menu);
        $this->createAttribute($request, $menu);

        return redirect()->to('menu/index')->withSuccess('Create menu successfly');
    }

    /**
    * Retrieve associated addon menu 
    * 
    * @param $request
    * @return void
    */
    private function createAddon($request, $menu)
    {
        if($request->addon_id !== null){
            foreach($request->addon_id as $a)
            {
                $menu->menuAddon()->firstOrCreate([
                    'addon_id' => $a,
                ]);
            }
        }
    }

    /**
    * Retrieve associated attribute menu 
    * 
    * @param $request
    * @return void
    */
    private function createAttribute($request, $menu)
    {
        if($request->attribute_id !== null){

            foreach($request->attribute as $key => $at)
            {
                $menu_attribute = $menu->menuAttribute()->firstOrCreate([
                    'attribute_id' => $key
                ]);

                foreach( $at as $attr ){
                        
                    $menu_attribute->attributeValues()->updateOrCreate([
                        'id' => @$attr['id']
                    ], [
                        'description' => $attr['description'],
                        'price' => $attr['price']
                    ]);

                }

            }
        }
    }


    /**
    * Upload image
    * 
    * @return void
    */
    private function uploadImage($request)
    {
        $extension = $request->file('image')->getClientOriginalExtension();

        return $request->file('image')->storeAs(
            'menu', $request->image->hashName(), 'public'
        );
    }

    /**
    * Menu edit
    *  
    * @param by id
    * @return View
    */
    public function edit($id)
    {
        $menus = Menu::with([
            'menuAddon',
            'menuAttribute' => function($q){
                $q->with(['option', 'attributeValues']);
            }
        ])->find($id);

        $menutypes = MenuType::get();
        $addons = Addon::get();
        $attributes = Attribute::get();
        $menuattributeValues = MenuAttributeValue::get();

        $menus->addons = $menus->menuAddon->pluck('addon_id')->toArray(); //FOR SELECT2 USE
        $menus->attributes = $menus->menuAttribute->pluck('attribute_id')->unique()->toArray(); //FOR SELECT2 USE
        
        return view('menu.edit', [
            'menus' => $menus,
            'menutypes' => $menutypes,
            'addons' => $addons,
            'attributes' => $attributes,
            'menuattributeValues' => $menuattributeValues
        ]);
    }

    /**
    * Menu Update
    * 
    * @param by id
    * @return View
    */
    public function update(Request $request, $id)
    {
        $id = $request->id;
        
        $input = $request->validate([
            'menu_type_id' => 'sometimes',
            'name' => 'required', 
            'short_description' => 'required', 
            'description' => 'required',
            'ingridients' => 'sometimes',
            'calories' => 'sometimes',
            'allergies' => 'sometimes',
            'image' => 'sometimes|image|max:5000',
            'price' => 'required',
            'discount' => 'sometimes',
            'discount_type' => 'sometimes',
            'available_time_starts' => 'sometimes',
            'available_time_ends' => 'sometimes',
            'status' => 'sometimes'
        ]);

        if ($request->image !== null) {
            $this->menuImageRemove($request->id);
        }

        if ($request->hasFile('image')) {
            $path = $this->uploadImage($request);
            $input['image'] = $path;
        }

        $menu = tap( Menu::find($id) )->update($input);
        
        $this->deleteOptions($request, $menu);
        $this->createAddon($request, $menu);
        $this->createAttribute($request, $menu);

        return redirect()->to('menu/index')->withSuccess('Update successfuly');
    }


    private function deleteOptions($request, $menu)
    {
        $attribute_values = collect();
        foreach ($request['attribute'] as $key => $attr) {
            $attribute_values = $attribute_values->merge( collect( $attr )->pluck('id') );
        }

        $menu->menuAddon()->whereNotIn('addon_id', $request->addon_id )->delete(); // DELETE ADDON
        $menu->menuAttribute()->whereNotIn('attribute_id', $request->attribute_id )->delete(); // DELETE ATTRIBUTE
        $menu->menuAttributeValue()->whereNotIn('menu_attribute_values.id', $attribute_values->toArray() )->delete(); // DELETE ATTRIBUTE VALUES
    }

    /**
    * Delete Menu image
    *
    * @param $id
    * @return void
    */
    private function menuImageRemove($id)
    {
        $menu = Menu::find($id);

        if ($menu->image !== null ) {
            unlink(storage_path() . '/app/public/' . $menu->image);
        }
    }

    /**
    * Menu delete
    * 
    * @param by id
    * @return view
    */
    public function delete($id)
    {
        $menus = Menu::find($id);

        if($menus->image !== null) {
            unlink(storage_path() . '/app/public/' . $menus->image);
        }
        $menus->delete();
     
        return redirect()->to('menu/index')->withSuccess('Deleted successfully');
    }

    /**
    * Status update active and inactive
    * 
    * @param by id
    * @return View 
    */
    public function statusUpdate($id, Request $request)
    {
        $menutype = Menu::find($id);
        $menutype->status = $request->status;
        $menutype->save();
  
        return response()->json(['data'=>'update successfully.']);
    }

    /**
    * Menu view
    * 
    * @param by id
    * @return View
    */
    public function view($id)
    {
        $menus = Menu::with('menuType')->find($id);
        $addons = Addon::get();

        return view('menu.view', [
            'menus' => $menus,
            'addons' => $addons
        ]);
    }

}
